/******************************************************************************************[Main.C]
MiniSat -- Copyright (c) 2003-2006, Niklas Een, Niklas Sorensson

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or
substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
**************************************************************************************************/
#include <omp.h>
#include <ctime>
 #include <cstring>
 #include <signal.h>
#ifdef _MSC_VER
 #include "C:\Users\youssefh\Desktop\minisat\minisatextended\zlib\include\zlib.h"
#else
 #include <zlib.h>
 #include <stdint.h>
 #include <ctime>
 #include <time.h>

 #include <cstring>
 #include <errno.h> 
#endif


#include "Solver.h"
/*************************************************************************************/
#ifdef _MSC_VER
#include <ctime>

static inline double cpuTime(void) {
    return (double)clock() / CLOCKS_PER_SEC; }
#else

#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>

static inline double cpuTime(void) {
  struct rusage ru;
    getrusage(RUSAGE_SELF, &ru);
    return (double)ru.ru_utime.tv_sec + (double)ru.ru_utime.tv_usec / 1000000; }
#endif


#if defined(__linux__)
static inline int memReadStat(int field)
{
    char    name[256];
    pid_t pid = getpid();
    sprintf(name, "/proc/%d/statm", pid);
    FILE*   in = fopen(name, "rb");
    if (in == NULL) return 0;
    int     value;
    for (; field >= 0; field--)
        fscanf(in, "%d", &value);
    fclose(in);
    return value;
}
static inline uint64_t memUsed() { return (uint64_t)memReadStat(0) * (uint64_t)getpagesize(); }


#elif defined(__FreeBSD__)
static inline uint64_t memUsed(void) {
    struct rusage ru;
    getrusage(RUSAGE_SELF, &ru);
    return ru.ru_maxrss*1024; }


#else
static inline uint64_t memUsed() { return 0; }
#endif

#if defined(__linux__)
#include <fpu_control.h>
#endif

//=================================================================================================
// DIMACS Parser:

#ifdef _MSC_VER
#define CHUNK_LIMIT 8192
#else
#define CHUNK_LIMIT 1048576
#endif

class StreamBuffer {
    gzFile  in;
    char    buf[CHUNK_LIMIT];
    int     pos;
    int     size;

    void assureLookahead() {
        if (pos >= size) {
            pos  = 0;
            size = gzread(in, buf, sizeof(buf)); } }
public:
    StreamBuffer(gzFile i) : in(i), pos(0), size(0) {
      assureLookahead(); }

    int  operator *  () { return (pos >= size) ? EOF : buf[pos]; }
    void operator ++ () { pos++; assureLookahead(); }
};

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
template<class B>
static void skipWhitespace(B& in) {
    while ((*in >= 9 && *in <= 13) || *in == 32)
        ++in; }

template<class B>
static void skipLine(B& in) {
    for (;;){
        if (*in == EOF || *in == '\0') return;
        if (*in == '\n') { ++in; return; }
        ++in; } }

template<class B>
static int parseInt(B& in) {
    int     val = 0;
    bool    neg = false;
    skipWhitespace(in);
    if      (*in == '-') neg = true, ++in;
    else if (*in == '+') ++in;
    if (*in < '0' || *in > '9') printf("PARSE ERROR! Unexpected char: %c\n", *in), exit(3);
    while (*in >= '0' && *in <= '9')
        val = val*10 + (*in - '0'),
        ++in;
    return neg ? -val : val; }

template<class B>
static void readClause(B& in, Cooperation* C, vec<Lit>& lits) {
    int     parsed_lit, var;
    lits.clear();
    for (;;){
        parsed_lit = parseInt(in);
        if (parsed_lit == 0) break;
        var = abs(parsed_lit)-1;
	int initvar = var;
	for(int t = 0; t < C->nbThreads;t++){
	  var = initvar;
	  while (var >= C->solvers[t].nVars()) C->solvers[t].newVar();
	}
        //while (var >= S.nVars()) S.newVar();
        lits.push( (parsed_lit > 0) ? Lit(var) : ~Lit(var) );
    }
}

template<class B>
static bool match(B& in, char* str) {
    for (; *str != 0; ++str, ++in)
        if (*str != *in)
            return false;
    return true;
}

template<class B>
static void parse_DIMACS_main(B& in, Cooperation* C) {
    vec<Lit> lits;
    for (;;){
        skipWhitespace(in);
        if (*in == EOF)
            break;
        else 
	   if (*in == 'p')
             {
              if (match(in, "p cnf"))
		{//warnings vars not used but necessary side effect.
                int vars    = parseInt(in);
                int clauses = parseInt(in);
		  //printf("c |  Number of variables:  %-12d                                         |\n", vars);
		  //printf("c |  Number of clauses:    %-12d                                         |\n", clauses);
             	}
	     else{
                printf("c PARSE ERROR! Unexpected char: %c\n", *in), exit(3);
               }
            } 
	   else 
		if (*in == 'c' || *in == 'p')
            		skipLine(in);
        	else{
			readClause(in, C, lits);
			for(int t = 0; t < C->nbThreads;t++)
		  		C->solvers[t].addClause(lits);
		}			
	} // for
}

// Inserts problem into solver.
//
static void parse_DIMACS(gzFile input_stream, Cooperation* C) {
    StreamBuffer in(input_stream);
    parse_DIMACS_main(in, C); }

//=================================================================================================

void printStats(Solver& solver)
{
    double   cpu_time = cpuTime();
    uint64_t mem_used = memUsed();
	printf("c threadId              : %i\n", solver.threadId);
    printf("c restarts              : %lu\n", solver.starts);
    printf("c conflicts             : %-12lu   (%.0f /sec)\n", solver.conflicts   , solver.conflicts   /cpu_time);
    printf("c decisions             : %-12lu   (%4.2f %% random) (%.0f /sec)\n", solver.decisions, (float)solver.rnd_decisions*100 / (float)solver.decisions, solver.decisions   /cpu_time);
    printf("c propagations          : %-12lu   (%.0f /sec)\n", solver.propagations, solver.propagations/cpu_time);
    printf("c conflict literals     : %-12lu   (%4.2f %% deleted)\n", solver.tot_literals, (solver.max_literals - solver.tot_literals)*100 / (double)solver.max_literals);
    if (mem_used != 0) printf("c Memory used           : %.2f MB\n", mem_used / 1048576.0);
    printf("c CPU time              : %g s\n", cpu_time);
}

//Solver* solver;
static void SIGINT_handler(int signum) {
  printf("s UNKNOWN\n");
  exit(0); 
}

//=================================================================================================
// Main:

void printUsage(char** argv)
{
  
    printf("USAGE: %s [options] <input-file> <result-output-file>\n\n  where input may be either in plain or gzipped DIMACS.\n\n", argv[0]);
    printf("OPTIONS:\n\n");
    printf("  -m n           : set n threads\n");
    printf("  -e l           : export clauses up to size l\n");
    printf("  -aimd p        : dynamic export v=1, throughput-based; v=2, throughput and quality-based, (0=static)\n");
    printf("  -polarity-mode = {true,false,rnd}\n");
    printf("  -decay         = <num> [ 0 - 1 ]\n");
    printf("  -rnd-freq      = <num> [ 0 - 1 ]\n");
    printf("  -verbosity     = {0,1,2}\n");
    printf("\n");
}

int getLimit(char *name) {
	char *value = getenv(name); 
	if (value == NULL) return 0;
	return atoi(value);
}


typedef enum { pre_none, pre_once, pre_repeat } preprocessMode;
char* Filename;

//time_t temps1;
int main(int argc, char** argv)
{
        int nbThreads = getLimit("MAXNBTHREAD");
	if((nbThreads == 0) || (nbThreads > 4)) nbThreads = 4;
	//printf("nbTh %d \n",nbThreads);
	//exit(0);
	int limitExport = 8;
	int aimd  = 0;
#if defined(__linux__)
    fpu_control_t oldcw, newcw;
    _FPU_GETCW(oldcw); newcw = (oldcw & ~_FPU_EXTENDED) | _FPU_DOUBLE; _FPU_SETCW(newcw);
    //    printf("c WARNING: for repeatability, setting FPU to use double precision\n");
#endif    

	//get Many flags
        int i, j;
	for (i = j = 0; i < argc; i++){
		if (strncmp(argv[i], "-m", 2) == 0){
			nbThreads = atoi(argv[++i]);
		}
		else 
		if (strncmp(argv[i], "-e", 2) == 0){
			limitExport = atoi(argv[++i]);
		}

		else
                  if (strncmp(argv[i], "-aimd", 2) == 0){
                    aimd = atoi(argv[++i]);
                  }		
	}
	
	Cooperation coop(nbThreads, limitExport, aimd);
	printf("c This is ManySAT 1.1 : %i thread(s) on %i core(s).\n", coop.nbThreads, omp_get_num_procs());
	 
	  
	for(int t = 0; t < nbThreads; t++){
		coop.solvers[t].threadId = t;
		coop.solvers[t].verbosity = 0;
	}

	for (i = j = 0; i < argc; i++){
		if (strncmp(argv[i], "-polarity-mode=", 15) == 0){
			if (strncmp(&argv[i][15], "true", 4) == 0)
				for(int t = 0; t < nbThreads; t++)
					coop.solvers[t].polarity_mode = Solver::polarity_true;
			else if (strncmp(&argv[i][15], "false", 5) == 0)
					for(int t = 0; t < nbThreads; t++)
						coop.solvers[t].polarity_mode = Solver::polarity_false;
			else if (strncmp(&argv[i][15], "rnd", 3) == 0)
					for(int t = 0; t < nbThreads; t++)
						coop.solvers[t].polarity_mode = Solver::polarity_rnd;
			else{
				printf("c error: unknown polarity-mode %s\n", &argv[i][15]);
				printUsage(argv);
				exit(0); }
		}else if (strncmp(argv[i], "-rnd-freq=", 10) == 0){
			double rnd = strtod(&argv[i][10], NULL);
			  /*if (rnd == 0){
				printf("c error: illegal rnd-freq constant %s\n", &argv[i][10]);
				printUsage(argv);
				exit(0); }
*/
			for(int t = 0; t < nbThreads; t++)
				coop.solvers[t].random_var_freq = rnd;
		}else if (strncmp(argv[i], "-decay=", 7) == 0){
			double decay = strtod(&argv[i][7], NULL);
			  /*if (decay == 0){
				printf("c error: illegal decay constant %s\n", &argv[i][7]);
				printUsage(argv);
				exit(0); }
*/
			for(int t = 0; t < nbThreads; t++)
				coop.solvers[t].var_decay = 1 / decay;
		}else if (strncmp(argv[i], "-verbosity=", 11) == 0){
			int verbosity = (int)strtol(&argv[i][11], NULL, 10);
			//if (verbosity == 0 && errno == EINVAL){
			  //printf("c error: illegal verbosity level %s\n", &argv[i][11]);
			  //printUsage(argv);
			//    exit(0); }
			for(int t = 0; t < nbThreads; t++)
				coop.solvers[t].verbosity = verbosity;
		}else if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "-help") == 0){
			printUsage(argv);
			exit(0);
		}else if (strncmp(argv[i], "-", 1) == 0){
			//printf("error: unknown flag %s\n", argv[i]);
			//printUsage(argv);
			//exit(0);
		}else
			argv[j++] = argv[i];
	}
    argc = j;

    if (argc == 2 && (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0))
        exit(0);


    Filename = argv[1];
    signal(SIGINT,SIGINT_handler);
    signal(SIGHUP,SIGINT_handler);


  double cpu_time = cpuTime();


  //temps1 = time(NULL);

    if (argc == 1)
        printf("c Reading from standard input... Use '-h' or '--help' for help.\n");

    gzFile in = (argc == 1) ? gzdopen(0, "rb") : gzopen(argv[1], "rb");
    if (in == NULL)
        printf("c ERROR! Could not open file: %s\n", argc == 1 ? "<stdin>" : argv[1]), exit(1);

    printf("c ============================[ Problem Statistics ]=============================\n");
    printf("c |                                                                             |\n");

 
    parse_DIMACS(in, &coop);
    
    gzclose(in);

    FILE* res = (argc >= 3) ? fopen(argv[2], "wb") : NULL;

    

    double parse_time = cpuTime() - cpu_time;
      //    printf("c |  Parsing time:         %-12.2f s                                       |\n", parse_time);

    
    if (!coop.solvers[0].simplify()){
        printf("c Solved by unit propagation\n");
	//#ifdef COMPETITION
        printf("s UNSATISFIABLE\n");
	  //#else
        if (res != NULL) fprintf(res, "UNSAT\n"), fclose(res);
        //printf("UNSATISFIABLE\n");
	//#endif
        exit(20);
    }
	for(int t = 1; t < nbThreads; t++)
		coop.solvers[t].simplify();

	int winner = 0;
	bool ret;
	bool result;
	omp_set_num_threads(nbThreads);
        //time_t temps2;
	//double tempstotal = 0;
#pragma omp parallel
{
	int t = omp_get_thread_num();
	coop.start = true; 
	ret = coop.solvers[t].solve(&coop);
	#pragma omp critical (accessToEnd) 
	{
	  if(coop.End == false){		
	    coop.End = true;
	    winner = t;
	    result = ret;
	  }
	}
}

	
 printStats(coop.solvers[winner]);
//  coop.print();

    printf("\nc --->m=%d, e=%d, aimd=%d, %s\ttime: %g\tconflicts: %-12lu ", coop.nbThreads, coop.limitExportClauses, coop.aimd, Filename, cpuTime() / coop.nbThreads, coop.solvers[winner].conflicts);
    printf("\n");

#ifdef COMPETITION
    //a virer
    printf(result ? "s SATISFIABLE\n" : "s UNSATISFIABLE\n");
    if (result){
        printf("v");
        for (int i = 0; i < S.nVars(); i++)
            if (S[winner].model[i] != l_Undef)
                printf(" %s%d", (S[winner].model[i]==l_True)?"":"-", i+1);
        printf(" 0\n");
    }
#else
      //    printf(result ? "SATISFIABLE\n" : "UNSATISFIABLE\n");
    if (res != NULL){
        if (result){
            fprintf(res, "SAT\n");
            for (int i = 0; i < coop.solvers[winner].nVars(); i++)
                if (coop.solvers[winner].model[i] != l_Undef)
                    fprintf(res, "%s%s%d", (i==0)?"":" ", (coop.solvers[winner].model[i]==l_True)?"":"-", i+1);
            fprintf(res, " 0\n");
        }else
	  fprintf(res, "UNSAT\n");
        fclose(res);
    }
    else{
      if(result){
            printf("s SATISFIABLE\n");
	    printf("v ");
            for (int i = 0; i < coop.solvers[winner].nVars(); i++)
                if (coop.solvers[winner].model[i] != l_Undef)
                    printf("%s%s%d", (i==0)?"":" ", (coop.solvers[winner].model[i]==l_True)?"":"-", i+1);
            printf(" 0\n");
        }else
            printf("s UNSATISFIABLE\n");

    }
#endif

    //#ifdef NDEBUG
    exit(result ? 10 : 20);     // (faster than "return", which will invoke the destructor for 'Solver')
    //#endif
}
